/******************************************************************************
    SG_Bargraph.c     A bargraph control that the user can touch.
    Copyright:        Open-Source, Jan2011, written by www.RomanBlack.com
    MCU:              P18F8527
    Dev.Board:        SmartGLCD (SmartGLCD 240x128 module)
    Oscillator:       HSPLL 32.0 MHz (8MHz xtal)
    Compiler:         MikroC v7.0 (or MikroC PRO 4.0)

  This demonstrates the GLCD and Touch Panel functions and draws a
  simple Bargraph control to the screen. A new function TP_convert_XY()
  is added, that converts TouchPanel XY ADC values to XY screen pixel coords. 
  The user can adjust the bargraph by touching it on the screen. It also
  displays XY ADC values on the GLCD and also screen pixel XY values.
******************************************************************************/
// Global Variables
unsigned char tp_x;     // holds results of Touch Panel adc values
unsigned char tp_y;
unsigned char tp_sx;    // holds Touch Panel screen pixel X and Y values
unsigned char tp_sy;
unsigned int tp_math;   // used for 16bit TP coords calcs

// defines used for the bargraph
#define BAR_X       70    // edit these to change the bar size and position
#define BAR_LENGTH  139   // (XY is top left corner of the bar)
#define BAR_Y       64
#define BAR_HEIGHT  16
unsigned char i;

unsigned char text[4];  // holds 3 digit number + NULL, for text display

// TPfast definitions
void TPfast_SetX(void);   // function to set X axis for next read
void TPfast_SetY(void);   // function to set X axis for next read
void TPfast_StartRead(void);      // function to start adc read from TouchPanel
unsigned char TPfast_Read(void);  // function to read adc result from TouchPanel

#include "T6963C_MikroC.h"      // include this if using MikroC (older)
//#include "T6963C_MikroC_PRO.h"  // include this if using MikroC PRO

#include "TPfast.c"     // my fast TouchPanel functions to read X and Y

//=============================================================================
//  TP CONVERT XY
//=============================================================================
void TP_convert_XY(void)
{
  //-------------------------------------------------------
  // The TouchPanel ADC XY values (0-255) are already in tp_x and tp_y.
  // This converts them to a calibrated screen pixel X Y from
  // 0-239 and 0-127 and stores them in variables tp_sx and tp_sy.
  // This is optimised for execution speed so you need to enter
  // in constants for your TouchPanel XY ADC values.
  // NOTE!! tp_math must be a 16bit variable; unsigned int tp_math
  // Conversion is done by simple math; Xpixels = (Xadc * 240 / Xadc_range)
  //-------------------------------------------------------
  #define TP_CAL_XMIN   12    // enter 4 ADC values for your screen pixel edges
  #define TP_CAL_XMAX   241   // (check them against displayed values!)
  #define TP_CAL_YMIN   21
  #define TP_CAL_YMAX   227
  #define TP_CAL_XRANGE (TP_CAL_XMAX - TP_CAL_XMIN)   // don't edit these 2
  #define TP_CAL_YRANGE (TP_CAL_YMAX - TP_CAL_YMIN)

  // Convert X adc value to screen pixels
  if(tp_x > TP_CAL_XMIN)            // make sure is >0 before doing calc!
  {
    tp_math = (tp_x - TP_CAL_XMIN);     // get X adc value - X min offset
    tp_math = (tp_math * 239);          // 0-239 pixels on X
    tp_math = (tp_math / TP_CAL_XRANGE);
    tp_sx = tp_math;
    if(tp_sx > 239) tp_sx = 239;
  }
  else tp_sx = 0;

  // Convert Y adc value to screen pixels
  if(tp_y > TP_CAL_YMIN)            // make sure is >0 before doing calc!
  {
    tp_math = (tp_y - TP_CAL_YMIN);     // get Y adc value - Y min offset
    tp_math = (tp_math * 127);          // 0-127 pixels on Y
    tp_math = (tp_math / TP_CAL_YRANGE);
    tp_sy = tp_math;
    if(tp_sy > 127) tp_sy = 127;
  }
  else tp_sy = 0;
  tp_sy = (127 - tp_sy);    // invert Y to match LCD pixel coords
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // setup PIC 18F8527 for SmartGLCD pins
  CMCON = 0x07;        // turn off comparators (make all pins digital)
  ADCON0 = 0b00000001;  // ADC module on
  ADCON1 = 0b00001101;  // AN0,AN1 are adc inputs, 0v-5v range
  ADCON2 = 0b00110010;  // ADC result left justified (0-255 range)

  LATA =  0b00000000;
  TRISA = 0b00000011;   // RA0,RA1 analog inputs (TP)
  LATC =  0b00000110;   // LEDs off at start
  TRISC = 0b00000000;   // C1, C2 backlight LED
  LATG =  0b00000001;   // LED off at start
  TRISG = 0b00000000;   // G0 backlight LED
  
  LATJ  = 0b00000000;   // RJ6=FS (1=font6 0=font8), RJ5=MD
  TRISJ = 0b00000000;   // GLCD control port

  BacklightRed    = 1;     // control the GLCD backlight leds; 0=on, 1=off
  BacklightGreen  = 0;
  BacklightBlue   = 1;

  //-------------------------------------------------------
  // startup delay, let the PSU voltage stabilise etc.
  Delay_ms(10);

  // Initialize T6963C GLCD
  //T6963C_init(240, 128, 8);   // init for MikroC PRO version
  T6963C_init(240, 128, 8, &PORTH, &PORTJ, 2, 1, 0, 4); // init for MikroC version
  T6963C_graphics(1);       // graphics mode = on
  T6963C_text(1);           // text mode = on (now both are on)
  T6963C_cursor(1);         // cursor and blink don't seem to work?
  T6963C_cursor_blink(1);

  //-------------------------------------------------------
  // draw stuff on GLCD that remains constant

  // write the text that will not change
  T6963C_Write_Text("SmartGLCD Bargraph Testing", 0, 0, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("Touch the Bar!", 0, 2, T6963C_ROM_MODE_OR);

  T6963C_Write_Text("TX=",  0, 14, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("TY=",  0, 15, T6963C_ROM_MODE_OR);

  T6963C_Write_Text("SX=", 10, 14, T6963C_ROM_MODE_OR);
  T6963C_Write_Text("SY=", 10, 15, T6963C_ROM_MODE_OR);

  // draw the bargraph bar outer frame
  T6963C_rectangle(BAR_X-2, BAR_Y-2, (BAR_X+BAR_LENGTH)+1, (BAR_Y+BAR_HEIGHT)+1, T6963C_WHITE);  
 
  //-------------------------------------------------------
  // main running loop (repeat 10 times a second)
  while(1)
  {
    //---------------------------------------------
    // get TouchPanel X value
    TPfast_SetX();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_x = TPfast_Read();

    // get TouchPanel Y value
    TPfast_SetY();       
    Delay_uS(500);
    TPfast_StartRead();   // start TP adc conversion
    tp_y = TPfast_Read();

    // format and display TP X ADC value
    ByteToStr(tp_x,text);
    T6963C_Write_Text(text, 3, 14, T6963C_ROM_MODE_OR);

    // format and display TP Y ADC value
    ByteToStr(tp_y,text);
    T6963C_Write_Text(text, 3, 15, T6963C_ROM_MODE_OR);

    //---------------------------------------------
    // get X Y coords as screen pixel coords; X=0-239 Y=0-127
    TP_convert_XY();  

    // format and display TP X pixel value
    ByteToStr(tp_sx,text);
    T6963C_Write_Text(text, 13, 14, T6963C_ROM_MODE_OR);

    // format and display TP Y pixel value
    ByteToStr(tp_sy,text);
    T6963C_Write_Text(text, 13, 15, T6963C_ROM_MODE_OR);

    //---------------------------------------------
    // now manually draw a bargraph bar

    // only draw the bar if the bar was touched!
    if(tp_sx>=(BAR_X-5) && tp_sx<=(BAR_X+BAR_LENGTH+5) &&
              tp_sy>=BAR_Y && tp_sy<=(BAR_Y+BAR_HEIGHT) )
    {          
      // draw horizontal bar as a sequence of rectangles.
      // this goes from left to right and always draws every part of the bar
      // but it draws them 2 different colours to make the bar "length" 
      for(i=BAR_X; i<(BAR_X+BAR_LENGTH); i+=4)
      {
        if(i < tp_sx)  T6963C_Rectangle(i,BAR_Y,i+2,(BAR_Y+BAR_HEIGHT-1),T6963C_WHITE);
        else           T6963C_Rectangle(i,BAR_Y,i+2,(BAR_Y+BAR_HEIGHT-1),T6963C_BLACK);
      }
    }
    //---------------------------------------------
    Delay_mS(100);
  }
}
//-----------------------------------------------------------------------------




