/******************************************************************************
  BaudTimer.c    Measures incoming baudrate of a serial 0x00 byte
  open-source  -  28rd Dec 2009  -  www.RomanBlack.com
  PIC 18F8520, 40MHz HSPLL, GLCD, BIGPIC6 dev board.

  This measures the time period for a 0x00 byte received in USART1.
  This period is for 9 bits, and period is displayed as "xxxx.x uS".
  Then the period is /9 and displayed on the GLCD to show actual
  incoming baudrate.
******************************************************************************/
unsigned char timermsb;
unsigned long math1;
unsigned long math2;
unsigned long math3;

unsigned char txt[16];

//=============================================================================
//   MAIN
//=============================================================================
void main()
{
	//--------------------------------------------------------
  // specific PIC18F8520 setup
  CMCON |= 0x07;              // turn off comparators
  ADCON1 |= 0x0D;             // Set AN0 and AN1 channel pins as analog
  //TRISB = 0b00001111;         // 4 buttons
  LATC  = 0b01000000;
  TRISC = 0b10000000;         // RC7=1, RC6=0 (usart1)
  MEMCON.EBDIS = 1;           // disable external memory bus

	//--------------------------------------------------------
  // GLCD setup
  Glcd_Init(&PORTJ, 0,1,2,3,5,4, &PORTD);   // pin setup on BIGPIC6
  Glcd_Fill(0x00);
  Glcd_Set_Font(Character8x8, 8, 8, 32);  
  GLCD_Write_Text("Baud Timer",(0*8),0,1);
  Glcd_Set_Font(FontSystem5x8, 5, 8, 32);    
  GLCD_Write_Text("Send 0x00 byte to",(0*6),1,1);
  GLCD_Write_Text("USART1 on RC7.",(0*6),2,1);

  // TMR1 setup
  T1CON = 0b00000000;   // TMR1 off, pre 1:1 (10 MHz)
  
	//--------------------------------------------------------
  // main loop
  while(1)
  {
    //-----------------------------------------------
    // just wait in a tight loop for a 0x00 byte to be sent
    TMR1L = 0;
    TMR1H = 0;
    PIR1.TMR1IF = 0;
    timermsb = 0;
    
    while(PORTC.F7);    // wait for start of serial in byte
    T1CON = 0x01;       // then turn TMR1 on

    while(!PORTC.F7)    // wait for 0x00 serial byte to end
    {
      if(PIR1.TMR1IF)      // count TMR1 rolls
      {
        PIR1.TMR1IF = 0;
        timermsb++;
      }
    }
    T1CON = 0;          // turn TMR1 off

    //-----------------------------------------------
    // now we have the captured period in 3 bytes
    math1 = timermsb;        // turn 3 bytes into 1 long, put in math1
    math1 = (math1 * 65536);
    math2 = TMR1H;
    math2 = (math2 * 256);
    math2 += TMR1L;
    math1 += math2; 
 
    // display period of the received byte in uS
    LongToStr(math1,txt);           // is in 0.1uS resolution
    txt[11] = txt[10];              // so move the dec place
    txt[10] = '.';
    txt[12] = 0;
    GLCD_Write_Text(txt,(0*6),4,1);
    GLCD_Write_Text("uS (9b)",(13*6),4,1);

    // convert from period to baud (actually is still baud*9)
    // this is done as 10 second/period for extra resolution digit
    // (do this division as a decrementing loop)
    math2 = 100000000;      // 100 million = 10 seconds
    math3 = 0;
    while(math2 >= math1)
    {
      math2 -= math1;
      math3++;
    }
    
    // now math3 equals 10sec/period*9, so fix the /9 first
    math3 = (math3 * 9);

    // and because it was 10 secs we get an extra digit resolution
    LongToStr(math3,txt);           // this is baudrate*10
    txt[11] = txt[10];              // so move the dec place
    txt[10] = '.';
    txt[12] = 0;
    GLCD_Write_Text(txt,(0*6),5,1); // and display as xxxxx.x baud
    GLCD_Write_Text("Baud",(13*6),5,1);
    
    // all done! just wait for another test byte to be sent
  }
}
//-----------------------------------------------------------------------------
