/******************************************************************************
  BaudrateConvert1.c   Dual USART bidirectional buffered baudrate converter!
  open-source  -  23rd Dec 2009  -  www.RomanBlack.com
  PIC 18F8520, 40MHz PLL, GLCD, BIGPIC6 dev board.

  This is a early application built with my crude multi-thread
  "PIC-thread loop frequency system". All tasks are performed in
  independent threads. For more info see; www.RomanBlack.com/PICthread.htm

  This project allows full asynchronous bidirectional exchange between
  2 USARTS at any baudrate, provided there are some pauses so the buffers
  do not overrun and that USART1 is the higher speed USART. The buffer
  usage is displayed on a 7 LED bargraph on PORTB.
******************************************************************************/

// thread engine variables
unsigned char loopnum;
unsigned char th3display;

// buffer vars
#define MAXBUFFER1 1792
unsigned char buffer1[MAXBUFFER1];
unsigned int inpointer1;
unsigned int outpointer1;
unsigned int bufferload1;
unsigned char i;

//=============================================================================
//   MAIN
//=============================================================================
void main()
{
	//----------------------------------------------------
  // specific PIC18F8520 setup
  CMCON |= 0x07;              // turn off comparators
  ADCON1 |= 0x0D;             // Set AN0 and AN1 channel pins as analog
  LATB = 0;
  TRISB = 0b00000000;         // 8 LEDs bargraph
  TRISC = 0b10000010;         // RC7=1, RC6=0 (usart1)
  TRISG = 0b00000100;         // RG2=1, RG1=0 (usart2)
  MEMCON.EBDIS = 1;           // disable external memory bus

	//----------------------------------------------------
  // setup USART1 - note! Must be the higher baudrate of the 2 usarts!
  SPBRG1 = 129; // "19200"  
  TXSTA1 = 0b00100100;    // BRGH = 1
  RCSTA1 = 0; 
  i = RCREG1;
  i = RCREG1;
  RCSTA1 = 0b10010000; 

  // setup USART2
  SPBRG2 = 129; // "4800"  
  TXSTA2 = 0b00100000;    // BRGH = 0
  RCSTA2 = 0;
  i = RCREG2;
  i = RCREG2;
  RCSTA2 = 0b10010000;

  // setup vars
  inpointer1 = 0;
  outpointer1 = 0;
  
	//----------------------------------------------------
  // main multi-thread engine loop
  while(1)
  {
    // LOOP SYNCHONISE ====================================
    // don't sync, just run loop at full speed
    loopnum++;

    // THREAD 0 ===========================================
    // HIGH PRIORITY!
    // check if usart1 received a byte, put it in buffer
    if(PIR1.RC1IF)
    {
      buffer1[inpointer1] = RCREG1;   // store the byte
      inpointer1++;
      if(inpointer1 >= MAXBUFFER1) inpointer1 = 0;
    }

    // THREAD 1 ===========================================
    // HIGH PRIORITY!
    // check if usart2 received a byte, send it out to usart1
    if(PIR3.RC2IF)
    {
      if(TXSTA1.TRMT) TXREG1 = RCREG2;  // if usart TX free, send byte
    }

    // THREAD 2 ===========================================
    // HIGH PRIORITY!
    // send usart1 byte if any bytes are in buffer
    bufferload1 = (inpointer1 - outpointer1); 
    if(bufferload1)
    {
      if(bufferload1 >= MAXBUFFER1) bufferload1 += MAXBUFFER1; // fix buffer roll
      th3display = (bufferload1 >> 8);  // get top byte for display
      
      if(TXSTA2.TRMT)   // if usart free, send another byte
      {
        TXREG2 = buffer1[outpointer1];
        outpointer1++;
        if(outpointer1 >= MAXBUFFER1) outpointer1 = 0;  // fix buffer roll
      }
    }
    else  th3display = 0;
    
    // THREAD 3 ===========================================
    // medium priority
    // display the buffer1 load as a 7 LED bargraph on PORTB
    if((loopnum & 0x0F) == 0)   // do this thread every 16th loop
    {
      LATB = 0;
      if(bufferload1)    LATB.F7 = 1;   // 7 LED bargraph
      if(th3display)     LATB.F6 = 1;
      if(th3display > 1) LATB.F5 = 1;
      if(th3display > 2) LATB.F4 = 1;
      if(th3display > 3) LATB.F3 = 1;
      if(th3display > 4) LATB.F2 = 1;
      if(th3display > 5) LATB.F1 = 1;
    }
    // THREAD END =========================================
  }
}
//-----------------------------------------------------------------------------

