/******************************************************************************
 * Project name:
     EP6_sig_gen.c   EasyPIC6 only, is a signal generator with variable PWM
 * Copyright:
     Open-source  -  Oct 2009  -  Roman Black
 * Description:
     Uses the EasyPIC6 development board as a fixed-freq PWM signal
     generator. There are 3 selectable freq. The PWM is fully adjustable
     in the 10bit range 0-1023. This code uses the MikroC PRO compiler
     libraries for the port expander IC and the COG LCD.
     The LCD displays Freq, duty cycle %, PWM value, PWM period in uS.

 * Test configuration:
     MCU:             PIC16F887
                      http://ww1.microchip.com/downloads/en/DeviceDoc/41291F.pdf
     Dev.Board:       EasyPIC6
                      http://www.mikroe.com/en/tools/easypic6/
     Oscillator:      HS, 8.0000 MHz
     Ext. Modules:    - freq signal comes in on PORTC.F0
     SW:              mikroC PRO for PIC v2.50
                      http://www.mikroe.com/en/compilers/mikroc/pro/pic/

******************************************************************************/

// global declarations

// Port Expander module connections
sbit  SPExpanderCS  at RA2_bit;
sbit  SPExpanderRST at RA3_bit;
sbit  SPExpanderCS_Direction  at TRISA2_bit;
sbit  SPExpanderRST_Direction at TRISA3_bit;
// End Port Expander module connections


// my vars
unsigned char freq_range;       // 0-2 (to set TMR2 prescale value)
unsigned char pwm_invert;       // to display pwm % as 20% or 80%

unsigned int freq;              // 0-65000, holds freq value to display
unsigned int pwm;               // 0-1023 holds 10bit PWM value
unsigned int pwm_percent;       // 0-99 to display pwm as %
unsigned int pwm_uS;            // 0-? to display pwm period in real uS

char txt[12];                   // used to display number string

#define XTAL_MHz  8     // set this to suit your xtal MHz
#define BASE_FREQ  ((XTAL_MHz * 250000) / 256)  // don't touch this

//=============================================================================

//-----------------------------------------------------------------------------
void display_data(void)
{
  //-------------------------------------------------------
  // get the freq number to display
  freq = BASE_FREQ;         // base PWM freq (to display)
  if(freq_range == 1) freq = (BASE_FREQ / 4);  // else if 1:4 prescaler
  if(freq_range == 2) freq = (BASE_FREQ / 16);      // if 1:16 prescaler

  // LCD display line 1
  // display the freq to display as "xxxxx Hz"
  WordToStr(freq,txt);
  SPI_Lcd_Out(1,1,txt);
  SPI_Lcd_Out(1,7,"Hz");

  // display the PWM as a percentage!
  // math - 0-1024 -> 0-100
  // therefore; % = (((pwm+5) *100) / 1024)
  // using *25 / 256 as it compiles much smaller and faster
  pwm_percent = (((pwm+5) * 25) / 256);
  if(!pwm_invert) WordToStr(pwm_percent,txt);
  else            WordToStr((100-pwm_percent),txt);
  SPI_Lcd_Out(1,12,txt+2);
  SPI_Lcd_Out(1,16,"%");

  // add high or low indicator for pwm duty display
  if(pwm_invert)  SPI_Lcd_Out(1,11,"H");
  else            SPI_Lcd_Out(1,11,"L");

  // LCD display line 2
  // display the pwm value to display as " xxxx PWM"
  WordToStr((1023-pwm),txt);        // invert PWM value for correct display
  SPI_Lcd_Out(2,2,txt+1);
  SPI_Lcd_Out(2,1,"P");

  // display the pwm period as uS (needs to be scaled by XTAL and freq_range)
  if(!pwm_invert)
  {
    pwm_uS = ((pwm * 64) / XTAL_MHz);  // default for TMR2 1:16 prescale
  }
  else
  {
    pwm_uS = (((1023-pwm) * 64) / XTAL_MHz);  // default for TMR2 1:16 prescale
  }
  // note! now the uS value is *4, but we have maxed the
  // resolution of our 16bit variable. we need to make it
  // *10 so we can display it as uS with one decimal place.
  // to get the *10 total, we do another operation of *2.5
  // which is done as *2 + 0.5 for simplicity.
  pwm_uS = (pwm_uS * 2) + (pwm_uS / 2);

  // now scale it for one of our 3 possible frequencies
  if(freq_range == 1) pwm_uS = (pwm_uS / 4);  // for 1:4 prescale
  if(freq_range == 0) pwm_uS = (pwm_uS / 16); // for 1:1 prescale

  // and format to text and display
  WordToStr(pwm_uS,txt);    // put the *10 value in txt string
  txt[5] = txt[4];          // move the decimal place digit
  txt[4] = '.';             // add the dec point char
  txt[6] = 0;               // add a null!
  if(txt[3] == ' ') txt[3] = '0';    // fix for 0.0 display
  SPI_Lcd_Out(2,8,txt);     // now display it!
  SPI_Lcd_Out(2,15,"uS");
}

//-----------------------------------------------------------------------------
void set_pwm(unsigned int pwm)
{
  //-------------------------------------------------------
  // 10bit pwm value is in pwm var
  // this funtion laods it into the PIC registers
  unsigned char pwm_trash;

  CCPR1L = (pwm >> 2);    // get 8 MSB bits
  pwm_trash = pwm;        // get bottom 8 bits of 10bit pwm
  CCP1CON &= 0b11001111;  // clear bottom 2 bits of 10bit pwm
  if(pwm_trash.F1) CCP1CON.F5 = 1;  // now load the bottom 2 bits!
  if(pwm_trash.F0) CCP1CON.F4 = 1;
}

//-----------------------------------------------------------------------------
void main()
{

  //-------------------------------------------------------
  // setup PIC 16F887 registers
  ANSEL  = 0;                               // Configure AN pins as digital
  ANSELH = 0;
  C1ON_bit = 0;                             // Disable comparators
  C2ON_bit = 0;

  TRISB = 0b11111111;
  TRISC = 0b00000000;         // PORTC.F2 (CCP1) is PWM output

  //-------------------------------------------------------
  // EasyPIC6 COG text LCD setup
  // Port Expander Library uses SPI1 module
  SPI1_Init();                              // Initialize SPI module used with PortExpander

  // show startup message
  SPI_Lcd_Config(0);                        // Initialize Lcd over SPI interface
  SPI_Lcd_Cmd(_LCD_CLEAR);                  // Clear display
  SPI_Lcd_Cmd(_LCD_CURSOR_OFF);             // Turn cursor off
  SPI_Lcd_Out(1,3, "EP6  Sig Gen");         // display startup text to Lcd
  //SPI_Lcd_Out(2,2, " ");
  Delay_1sec();
  Delay_1sec();
  
  // clear LCD again before main
  SPI_Lcd_Cmd(_LCD_CLEAR);

  //-------------------------------------------------------
  // setup the timers

  // setup TMR2 to generate the main frequency
  T2CON = 0b00000100;   // TMR2 ON, precale 1:1, 8Mhz xtal = 2MHz
  PR2 = 0xFF;           // TMR2 counts 0-255 (256)

  CCP1CON = 0b00001111;   // PWM mode, single output on P1A

  // setup my vars
  freq_range = 0;
  pwm_invert = 1;
  
  pwm = 512;
  set_pwm(pwm);

  //-------------------------------------------------------

  // draw the freq and pwm data to LCD
  display_data();

  //-------------------------------------------------------
  // now do the main run loop
  while(1)
  {                                
    // note! PWM is generated automatically by the CCP1 module
  
    // debounce delay sets button rate
    Delay_ms(200);
  
    // toggle pwm display invert feature
    if(PORTB.F7)  // if but pressed
    {
      pwm_invert.F0 = ~pwm_invert.F0;
      display_data();
    }

    // change freq range
    if(PORTB.F6)  // if but pressed
    {
      freq_range++;
      if(freq_range > 2) freq_range = 0;
      if(freq_range == 0) T2CON = 0b00000100;   // 1:1
      if(freq_range == 1) T2CON = 0b00000101;   // 1:4
      if(freq_range == 2) T2CON = 0b00000110;   // 1:16
      display_data();
    }

    // change pwm
    if(PORTB.F2)  // if but pressed
    {
      if(pwm > 0) pwm--;
      set_pwm(pwm);
      display_data();
    }
    if(PORTB.F3)  // if but pressed
    {
      pwm++;
      if(pwm > 1023) pwm = 1023;
      set_pwm(pwm);
      display_data();
    }
    if(PORTB.F4)  // if but pressed
    {
      if(pwm >= 20) pwm -= 20;
      else pwm = 0;
      set_pwm(pwm);
      display_data();
    }
    if(PORTB.F5)  // if but pressed
    {
      pwm += 20;
      if(pwm > 1023) pwm = 1023;
      set_pwm(pwm);
      display_data();
    }
  }
}
