/******************************************************************************
  DecimalFreqLF.c  Generates exact decimal frequency on a PIC using any xtal.
  Copyright; Sep 2012 Roman Black
  Compiler; MikroC or MikroC PRO
  Open source; use it for whatever you like, but please mention me as the
  original author and mention my web page; www.RomanBlack.com/High_Acc_Timing.htm

  Description; This generates a frequency output by toggling a PIC output pin
  based on a mathematical relationship, set by one simple variable; freqHz.
  The freq is in real decimal Hz, in a resolution of 0.00001 Hz, so you can
  make very precise frequencies like 23.13447 Hz. Also it has the advantage of
  working with any xtal value, just set your xtal details in the #define below.
  More details on how this works can be found in the "how_works.txt" file
  supplied with this project, and also on my web page.

  Note! Make sure with PIC16 that total interrupt size is less than 90
  instructions (check MikroC View Statistics). You may need to change the
  MikroC optimiser settings. (With PIC18 the interrupt will be about
  55 instructions so no problems there.) Also the TMR2 interrupt should be
  the only interrupt used in the project.

  This version is in 0.00001 Hz resolution, and is limited to osc <= 16MHz
  Minimum frequency; 0.00001 Hz
  Maximum frequency; 20000.00000 Hz (when using 16MHz osc)
  (Max freq is calculated by; TMR2 freq / 200)

  Compatibility; PIC16 and PIC18 both ok, requires TMR2.
******************************************************************************/
// set this number to your XTAL or HSPLL MHz value (same as MikroC project osc)
// note! in this version the osc must be <= 16000000
#define HZ 16000000   // 16MHz HS xtal = 16000000

// (Don't change this define! See "how_works.txt" for details.)
#define TOGGLE (HZ * 125)

// RAM vars
volatile unsigned long accum;
volatile unsigned long freqHz;


//=============================================================================
//   INTERRUPT
//=============================================================================
void interrupt(void)
{
  //-------------------------------------------------------
  // This is the TMR2 interrupt, set to 100 timer ticks.
  // Frequency generation is done by a combination of DDS accumulator
  // and a decimal bresenham subtraction that makes the value
  // in actual decimal fractions of Hz.
  // For more info see my page; www.RomanBlack.com/High_Acc_Timing.htm
  //-------------------------------------------------------
  PIR1.TMR2IF = 0;      // clear TMR2 roll flag
  accum += freqHz;      // decimal DDS addition into accumulator
  if(accum >= TOGGLE)
  {
    PORTB = (PORTB ^ 0x01);   // toggle output pin to make frequency
    accum -= TOGGLE;    // decimal bresenham subtraction, keep remainder
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//   MAIN
//=============================================================================
void main(void)
{
  //-------------------------------------------------------
  // setup PIC 16F628A
  CMCON = 0x07;   // comparators OFF

  PORTA = 0b00000000;   // PORTA all outputs
  TRISA = 0b00000000;

  PORTB = 0b00000000;   // RB0 is freq output, rest of PORTB also outputs
  TRISB = 0b00000000;

  // setup TMR2 to roll every 200 ticks
  T2CON = 0b00000100;   // TMR2 ON, 1:1 prescale
  PR2 = (100-1);        // TMR2 int occurs every 100 ticks

  // enable interrupt here (and just leave TNR2 int running in this example)
  INTCON = 0b11000000;  // GIE = ON, PEIE = ON
  PIE1.TMR2IE = 1;      // TMR2 interrupt ON
  
  // To set the frequency just set the freqHz variable.
  // If working with low frequencies it may be good to
  // clear the accumulator. Also, start the interrupt if needed.
  // Note! This variable is in Hz*100000, so 23Hz = 2300000
  freqHz = 2312345;  // 23.12345 Hz
  accum = 0;

  //-------------------------------------------------------
  // This is the main running loop/
  // To turn the output freq ON, enable TMR2 interrupt; (PIE1.TMR2IE = 1;)
  // To turn the output freq OFF, disable TMR2 interrupt; (PIE1.TMR2IE = 0;)
  while(1)
  {
    //---------------------------------------------
    // your code goes here!



  }
}
//-----------------------------------------------------------------------------












