/******************************************************************************
  SH1_Temp.c   Temperature controller using LM335 temp sensor
  Open-source  -  5th Nov 2009  -  Roman Black
    www.RomanBlack.com/shift1/sh1_projects.htm

  This is a complete temperature controller. The temperature
  control "Setpoint" can be set anywhere from -40'C to +100'C.
  (This is the operating range of the LM335 temp sensor)
  It can be used for incubators, room heaters, freezers etc.

  Two buttons control the setpoint temperature; UP and DOWN
  Hold these to adjust the setpoint.
  
  The LCD shows the actual temperature on the top line in 'C
  (The top line also has a "pulse" indicator to show it is operating)
  The bottom line shows HEAT/COOL and the Setpoint temperature

  PIC pins;
   GP0 = ADC in, connects to Vout pin of LM335 temperature sensor
   GP1 = button (lo = pressed) adjust Setpoint UP
   GP2 = digital output to Shift1-LCD
   GP3 = button (lo = pressed) adjust Setpoint DOWN (needs 10k pullup resistor!)
   GP4 = HEAT control output; HI when temp < setpoint
   GP5 = HEAT control output (inverted); LO when temp < setpoint

******************************************************************************/

// global variables;

signed int set_temp absolute 0x20;       // setpoint voltage 
unsigned char set_tempL absolute 0x20;   // overload vars for smaller code
unsigned char set_tempH absolute 0x21;

signed int adc_temp;           // ADC "voltage" 10bit value

unsigned int utemp;            // used for temp calcs

unsigned char buttons;         // buffer for buttons
unsigned char adj_res;         // adjust button change resolution
unsigned char pulse;           // for timing the display pulse indicator
unsigned char load_timer;      // for timing a minimum load ON period

char txt[9];                    // used to display number string

#define EE_SETL   0x00          // EEPROM addresses for stored data;
#define EE_SETH   0x01          // (setpoint lo and hi bytes)

// function declarations;
void message1(void);      // text messages to draw to LCD
void message2(void);

//=============================================================================


//-----------------------------------------------------------------------------
// this line adds the Shift1 and LCD functions
#include "Shift1_LCD.c"
//-----------------------------------------------------------------------------


// wrap delay as a function call to save code size
void delay_500ms(void)
{
  Delay_mS(460);  // this value gives about 1 second per "pulse"
}


//=============================================================================
//  TEST TEMPERATURE
//=============================================================================
void test_temperature(void)
{
  //-------------------------------------------------------
  // test temperature procedure;
  //  1. read the ADC voltage from the LM335 temp sensor
  //  2. convert the ADC reading into degrees Kelvin
  //  3. subtract a calibration offset, this turns 'K into 'C
  //-------------------------------------------------------

  // read voltage from AN0 (RA0) which is the LM335 temperature sensor.
  ADCON0.GO = 1;      // start ADC conversion
  while(ADCON0.GO);   // wait for conversion to be finished

  utemp = ADRESH;      // put the 10 bit ADC value in signed 16bit var
  utemp = (utemp * 256);
  utemp += ADRESL;  

  // convert the ADC reading into 'K
  // each ADC count is = 0.5' so multiply by 5, this gives 'K *10
  // (so it adds one decimal place)
  utemp = (utemp << 2) + utemp;  // adc_temp = (adc_temp * 5)
  adc_temp = utemp;   // copy to a signed variable now
  
  // now convert from 'K to 'C by subtracting a cailbration
  // value. With a perfect LM335 and 5.00v supply, the value
  // at 25'C is; 1024/5.00v*2.98v = 610 ADC counts = (x5) 3050 
  // which needs to be 25'C or 250, so (3050-250) = 2800 cal value.
  // My 5v supply is actual 5.07v (fairly typical) so thats;
  // 1024/5.07*2.98 = 602 ADC = (x5) 3010; (3010-250) = 2760 cal value.

  // note! you can also adjust this (5 counts = 0.5'C)  to "trim"
  // the LM335. Just test displayed temp against a proper thermometer.
  // My test LM335 reads about 0.4'C low, so; (2760-5) = 2755
  adc_temp -= 2755;    // subtract the calibration value
}
//-----------------------------------------------------------------------------


//=============================================================================
//  FORMAT TEMPERATURE
//=============================================================================
void format_temperature(signed int ftemp)
{
  //-------------------------------------------------------
  // formats the temperature value into a text string
  // so it can be diplayed on LCD.
  // initially temp is a number; [- nnnn]
  // we format that to show;     [-nnn.n'C]
  //-------------------------------------------------------

  // convert 4 digit number to text string
  inttostr(ftemp,txt);   

  // format the data within the text string
  txt[1] = txt[2];    // move 3 digits, 1 place to the left
  txt[2] = txt[3];  
  txt[3] = txt[4];  
  txt[4] = '.';       // add dec point
  txt[6] = 0xDF;      // add 'C chars
  txt[7] = 'C';       
  txt[8] = 0;         // alaways add NULL

  // and correct for 0.x and -0.x situations
  if(txt[3]==' ') txt[3] = '0';  // change  .n to 0.n
  if(txt[3]=='-')                // change -.n to -0.n
  {
    txt[2] = '-'; 
    txt[3] = '0'; 
  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//   MAIN
//=============================================================================
void main()
{
  //-------------------------------------------------------
  // 12F675 setup registers etc
  
  ANSEL =  0b00100001;   // ADC fosc32, AN0 is analog input
  ADCON0 = 0b10000001;   // right justified, VREF=Vdd, ADC=AN0, ADC is ON

  CMCON = 0x07;   // comparators OFF

  TRISIO = 0b00001011;    // GP2 out, GP1,3 buttons, GP0 is ADC in
  GPIO =   0b00100100;    // GP2 normally HI, goes to Shift1-LCD
  WPU =    0b00000010;    // pin pullups; 1 = pullup ON

  //-------------------------------------------------------
  // setup timers
  // TMR0 is a free running timer, at 1:1 (8MHz xtal = 2MHz)
  OPTION_REG = 0b00001000;  // pin pullups enabled, TMR0 = 1:1 prescale

  //-------------------------------------------------------
  // small delay for PSU to stabilise
  delay_500ms();

  // get the previous setpoint variables from PIC internal eeprom
  set_tempL = eeprom_read(EE_SETL);  
  set_tempH = eeprom_read(EE_SETH);  

  // check and fix invalid set_temp (like if eeprom was blank!)
  if(set_tempH == 0xFF)  
  {
    set_temp = 400;                  // use default set_temp of +40'C
    eeprom_write(EE_SETL,set_tempL); // and save +40'C to eeprom too
    eeprom_write(EE_SETH,set_tempH);
  }

  // setup any other variables
  load_timer = 0;
  adj_res = 0;
  
  //-------------------------------------------------------
  // initialise the LCD
  SH1_Lcd_Init();
  SH1_lcd_backlight = 0;  // LCD backlight is OFF

  // show startup message
  SH1_Lcd_Move(0,4); // start of line0
  message1();     // "Temp"

  SH1_Lcd_Move(1,5); 
  message2();     // "Set"

  delay_500ms();
  delay_500ms();

  //-------------------------------------------------------
  // TEMP!
  // display the setpoint volts
  format_temperature(set_temp);  // format by range, add dec point etc
  SH1_Lcd_Out(1,8,txt);       // display it


  //-------------------------------------------------------
  // now main run loop here
  while(1)
  {
    //---------------------------------------------
    // loop executes roughly every 0.5 sec
    delay_500ms();

    //---------------------------------------------
    // read and display battery voltage
    test_temperature();             // read volts from adc
    format_temperature(adc_temp);   // format by range, add dec point etc
    SH1_Lcd_Out(0,8,txt);       // display it

    // and display the pulse to show unit is operating
    pulse++;
    SH1_Lcd_Move(0,1); 
    if(pulse.F0) SH1_Lcd_Char('*');
    else         SH1_Lcd_Char(' ');
    
    //---------------------------------------------
    // now control the HEAT and display HEAT message
    // load_timer is set to 6 (=3 seconds)
    if(adc_temp < set_temp)  load_timer = 6;
    
    // display "HEAT" message on LCD if heat is ON
    if(load_timer)
    {
      GPIO.F4 = 1;        // actual HEAT ON
      GPIO.F5 = 0;

      SH1_Lcd_Move(1,0);  // HEAT message
      SH1_Lcd_Char('H');
      SH1_Lcd_Char('E');
      SH1_Lcd_Char('A');
      SH1_Lcd_Char('T');
      load_timer--; // gives a load ON minimum period
    }
    else     // else is COOL
    {
      GPIO.F4 = 0;        // actual HEAT OFF
      GPIO.F5 = 1;

      SH1_Lcd_Move(1,0);  // clear HEAT message
      SH1_Lcd_Char('C');
      SH1_Lcd_Char('O');
      SH1_Lcd_Char('O');
      SH1_Lcd_Char('L');
    }      
    //---------------------------------------------
    // test the 2 buttons and adjust the temperature setpoint
    // temperature setpoint is constrained by the
    // LM335 temperature limts; -40'C to +100'C
    
    if(!GPIO.F1)    // setpoint UP button pressed
    {
      if(!adj_res)
      {
        if(set_temp < 1000-5) set_temp+=5;   // 100'C  step = 0.5'C
      }
      else
      {
        if(set_temp < 1000-25) set_temp+=25; // 100'C  step = 2.5'C
      }
      adj_res = 1;                      // speed buttons up
      format_temperature(set_temp); 
      SH1_Lcd_Out(1,8,txt);             // display new settemp
      eeprom_write(EE_SETL,set_tempL);  // save value to eeprom too!
      eeprom_write(EE_SETH,set_tempH);
    }
    if(!GPIO.F3)    // setpoint DOWN button pressed
    {
      if(!adj_res)
      {
        if(set_temp >= -40 +5)  set_temp -= 5;  // -40'C step = 0.5'C
      }
      else
      {
        if(set_temp >= -40 +25) set_temp -= 25; // -40'C step = 2.5'C
      }
      adj_res = 1;                      // speed buttons up
      format_temperature(set_temp); 
      SH1_Lcd_Out(1,8,txt);             // display new settemp
      eeprom_write(EE_SETL,set_tempL);  // save value to eeprom too!
      eeprom_write(EE_SETH,set_tempH);
    }  
    // slow the buttons again once both are released
    if(GPIO.F1 && GPIO.F3) adj_res = 0;
    //---------------------------------------------

  }
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MESSAGES     LCD text messages, done like this to save some RAM 
//=============================================================================
void message1(void)
{
  SH1_Lcd_Char('T');
  SH1_Lcd_Char('e');
  SH1_Lcd_Char('m');
  SH1_Lcd_Char('p');
}
void message2(void)
{
  SH1_Lcd_Char('S');
  SH1_Lcd_Char('e');
  SH1_Lcd_Char('t');
}

//-----------------------------------------------------------------------------



